classdef vehConf
    %conf = defaultConf(veh_type)
    % Specify default component names
    %
    % Input arguments
    % ---------------
    % veh_type : char
    %   Vehicle type. Supported types are: "conv", "bev".
    %
    % Outputs
    % ---------------
    % conf : struct
    %   Structure whose fields specify names for each component. Pass this
    %   structure as input to createConfiguration to create the vehicle
    %   configuration.

    properties
        vehicle_type {mustBeMember(vehicle_type, ["conv", "bev"])} = "conv";
        vehicle_name {mustBeTextScalar} = "";
        finalDrive_name {mustBeTextScalar} = "";
        engine_name {mustBeTextScalar} = "";
        transmission_name {mustBeTextScalar} = "";
        em_name {mustBeTextScalar} = "";
        battery_name {mustBeTextScalar} = "";
        dynCharger_name {mustBeTextScalar} = "";
        supercap_name {mustBeTextScalar} = "";
    end

    methods

        function conf = vehConf(veh_type)

            conf.vehicle_type = veh_type;
            conf.finalDrive_name = "3_62";
            % Assign default components names
            conf.vehicle_name = "500e";
            switch veh_type
                case "conv"
                    conf.engine_name = "Toyota_TNGA";
                    conf.transmission_name = "sixSpeed";
                case "bev"
                    conf.finalDrive_name = "3_62";
                    conf.em_name = "ldv125";
                    conf.battery_name = "NMC";
                    conf.transmission_name = "singleSpeed";
            end

        end

        % Set/get methods
        function conf = set.vehicle_name(conf, name)
            name = conf.checkComponentName(name, "vehicle_body");
            conf.vehicle_name = name;
        end
        function conf = set.finalDrive_name(conf, name)
            name = conf.checkComponentName(name, "final_drive");
            conf.finalDrive_name = name;
        end
        function conf = set.engine_name(conf, name)
            name = conf.checkComponentName(name, "engine");
            conf.engine_name = name;
        end
        function conf = set.transmission_name(conf, name)
            name = conf.checkComponentName(name, "transmission");
            conf.transmission_name = name;
        end
        function conf = set.em_name(conf, name)
            name = conf.checkComponentName(name, "e-machine");
            conf.em_name = name;
        end
        function conf = set.battery_name(conf, name)
            name = conf.checkComponentName(name, "battery");
            conf.battery_name = name;
        end
        function conf = set.dynCharger_name(conf, name)
            name = conf.checkComponentName(name, "dynamic_charger");
            conf.dynCharger_name = name;
        end
        function conf = set.supercap_name(conf, name)
            name = conf.checkComponentName(name, "supercapacitor");
            conf.supercap_name = name;
        end
    end
    methods (Static)
        function name = checkComponentName(name, folderName)
            % Append .mat extension if absent
            [~, name, ext] = fileparts( name );
            if isempty(ext) || any(strcmp(ext, ["", ".mat"]))
                filename = name + ".mat";
            else
                error("Unrecognized file extension. " + name + " should point to a .mat file.")
            end
            % Check if the specified component exists in the library
            filename = fullfile("library", folderName, filename);
            if ~isfile( filename )
                warning( "Component " + name + " was not found in the library/" + folderName + " folder." )
            end
        end
    end

end



